/**
 * 将ArrayBuffer转换成16进制字符串
 * @param buffer
 * @returns {string}
 */
 export function ab2hex(buffer) {
  var hexArr = Array.prototype.map.call(
    new Uint8Array(buffer),
    function (bit) {
      return ('00' + bit.toString(16)).slice(-2)
    }
  )
  return hexArr.join('');
}

/**
 * 16进制转字符串
 * @param hexCharCodeStr
 * @returns {string}
 */
 export function hexCharCodeToStr(hexCharCodeStr) {
  var trimedStr = hexCharCodeStr.trim();
  var rawStr =
    trimedStr.substr(0,2).toLowerCase() === "0x"
      ?
      trimedStr.substr(2)
      :
      trimedStr;
  var len = rawStr.length;
  if(len % 2 !== 0) {
    alert("Illegal Format ASCII Code!");
    return "";
  }
  var curCharCode;
  var resultStr = [];
  for(var i = 0; i < len;i = i + 2) {
    curCharCode = parseInt(rawStr.substr(i, 2), 16); // ASCII Code Value
    resultStr.push(String.fromCharCode(curCharCode));
  }
  return resultStr.join("");
}

/**
 * 将16进制字符串转换成ArrayBufer
 * @param str
 * @returns {*}
 */
 export function hex2buffer(str) {
  let val = ""
  if(!str) return;
  let length = str.length;
  let index = 0;
  let array = []
  while(index < length){
    array.push(str.substring(index,index+2));
    index = index + 2;
  }
  val = array.join(",");
  // 将16进制转化为ArrayBuffer
  return new Uint8Array(val.match(/[\da-f]{2}/gi).map(function (h) {
    return parseInt(h, 16)
  })).buffer
}

/**
 * 开启蓝牙适配器
 * @returns {Promise<unknown>}
 * @constructor
 */
 export function openBLEAdapter() {
  return new Promise((resolve, reject) => {
    wx.openBluetoothAdapter({
      // 调用微信小程序api 打开蓝牙适配器接口
      success: function (res) {
        console.log('初始化蓝牙适配器成功 res=', res)
        // 通知蓝牙模块已启用成功
        resolve(true);
      },
      fail: function (res) {
        // 如果手机上的蓝牙没有打开，可以提醒用户
        wx.showModal({
          title: '提示',
          content: '请开启手机蓝牙！',
          showCancel: false,
          success: function (res) {
            if (res.confirm) {
              console.log('用户点击确定');
            }
          }
        })
        resolve(false)
      }
    })
  })
}

/**
 * 扫描蓝牙设备
 * @param servicesUUIDs 需要过滤的设备
 */
export function scanBluetoothDevices(servicesUUIDs) {
  console.log('扫描蓝牙设备 servicesUUIDs=', servicesUUIDs)
  return new Promise((resolve, reject) => {
      wx.startBluetoothDevicesDiscovery({
          allowDuplicatesKey: false,
          interval: 0,
          services: servicesUUIDs, // 如果填写了此UUID，那么只会搜索出含有这个UUID的设备，建议一开始先不填写或者注释掉这一句
          success: function (res) {
            // 已扫描完毕
            resolve({ status: 0, res: res })
          },
          fail: function (err) {
            // 如果手机上的蓝牙没有打开，可以提醒用户
            console.log(err)
            wx.showModal({
              title: '搜索失败',
              content: err.errMsg,
              showCancel: false,
              success: function (res) {
                if (res.confirm) {
                }
              }
            })

            reject(err)
          }
      })
  })
}

/**
 * 关闭蓝牙搜索
 */
export function stopBluetoothDevicesDiscovery() {
  // 连接蓝牙成功之后关闭蓝牙搜索
  wx.stopBluetoothDevicesDiscovery({
    success: function (res) {
      console.log('已关闭蓝牙搜索', res);
    }
  })
}

/**
 * 获取已经扫描到的设备列表
 * @param scanCodeMac
 * @returns {Promise<unknown>}
 */
export function getBluetoothDevicesList(isEnableFilter = true) {
  return new Promise((resolve, reject) => {
    wx.getBluetoothDevices({
      success: (res) => {
        if (res.errMsg != 'getBluetoothDevices:ok') return

        const deviceList = []
        // 获取到一个或多个设备
        const len = res.devices.length
        console.log(`getBluetoothDevicesList 获取到设备 ${len} 个`)

        for (let i = 0; i < len; i++) {
          const device = res.devices[i]
          if (device.advertisData) {
            let advertisData = ab2hex(device.advertisData); // 获取广播数据包
            if (advertisData && advertisData.length >= 26) {
              let deviceMac = advertisData.substring(0, 12).toUpperCase() // 转大写
              let deviceName = hexCharCodeToStr(advertisData.substring(12, 26));
              device.filterName = advertisData.substring(12, 26) // 626c654c6f636b == bleLock
              device.sn = hexCharCodeToStr(advertisData.substring(26, 42));
              device.mac = device.mac || deviceMac;
              device.deviceName = deviceName

              // 过滤设备
              if (isEnableFilter) {
                if (device.localName && deviceName == 'bleLock') {
                  deviceList.push(device)
                }
              } else {
                deviceList.push(device)
              }
            }
          }
        }

        resolve(deviceList)
      },
      fail: function(){
        console.log("搜索蓝牙设备失败")
        reject([])
      }
    })
  })
}

/**
 * 连接指定的蓝牙设备
 * 获取到设备之后连接蓝牙设备
 * 连接低功耗蓝牙设备。
 * 若APP在之前已有搜索过某个蓝牙设备，并成功建立连接，可直接传入之前搜索获取的 deviceId 直接尝试连接该设备，无需进行搜索操作。
 *
 * @param deviceId  需要连接的设备ID
 */
export function createBLEConnection(deviceId) {
  console.log('连接指定的蓝牙设备', deviceId)

  return new Promise((resolve, reject) => {
    wx.createBLEConnection({
      // 这里的 deviceId 需要已经通过 createBLEConnection 与对应设备建立链接
      deviceId: deviceId,//设备id
      success: function (res) {
        console.log("连接蓝牙成功!")
        //连接蓝牙成功之后关闭蓝牙搜索
        stopBluetoothDevicesDiscovery()
        resolve(res)
      },
      fail: function (err) {
        console.log(`createBLEConnection err`, err)
        reject(err)
      }
    })
  })
}

/**
 * 获取BLE设备ServiceId
 * @param serviceUUID
 * @param deviceId
 * @returns {Promise<unknown>}
 */
 export function getBLEDeviceServices(serviceUUID = 'FFE0', deviceId) {
  console.log('获取指定蓝牙设备所有服务', 'serviceUUID=',serviceUUID, 'deviceId=', deviceId)
  return new Promise((resolve, reject) => {
    wx.getBLEDeviceServices({
      // 这里的 deviceId 需要已经通过 createBLEConnection 与对应设备建立链接
      deviceId: deviceId,
      success: function (res) {
        console.log("getBLEDeviceServices=", res)
        const services = {
          serviceId: null
        }
        //查找主服务，并且UUID包含FFE0
        res.services.forEach((item) => {
          if (item.isPrimary && item.uuid.indexOf(serviceUUID) != -1) {

            // 获取蓝牙设备某个服务中所有特征值
            services.serviceId = item.uuid
          }
        })
        resolve(services)
      },
      fail: function(err) {
        reject(err)
      }
    })
  })
}

/**
 * 获取蓝牙设备某个服务中所有特征值
 * @param deviceId
 * @param serviceId
 * @returns {Promise<unknown>}
 */
 export function getBLEDeviceCharacteristics(deviceId, serviceId) {
  console.log('获取蓝牙设备某个服务中所有特征值', 'deviceId=', deviceId, 'serviceId=',serviceId)
  return new Promise((resolve, reject) => {
    wx.getBLEDeviceCharacteristics({
      // 这里的 deviceId 需要已经通过 createBLEConnection 与对应设备建立链接
      deviceId: deviceId,
      // 这里的 serviceId 需要在上面的 getBLEDeviceServices 接口中获取
      serviceId: serviceId,
      success: function (res) {
        console.log("获取蓝牙设备某个服务中所有特征值：res=", res)
        console.log("deviceId="+deviceId+",serviceId="+serviceId+",特征值,共=", res.characteristics.length, "个")

        // 循环所有特征值,一般我们只需要notify通知，write写 这两个
        const characteristics = {
          notifyCharacteristicsId: null,
          writeCharacteristicsId: null
        }

        for (let i = 0; i < res.characteristics.length; i++) {
          const model = res.characteristics[i]
          if (model.properties.notify == true) {
            console.log("通知notify=", model.uuid);
            characteristics.notifyCharacteristicsId = model.uuid
          }
          if (model.properties.write == true){
            console.log("可写write=", model.uuid)
            characteristics.writeCharacteristicsId = model.uuid
          }
        }

        resolve(characteristics)
      },
      fail: function(err) {
        reject(err)
      }
    })
  })
}

/**
 * 订阅蓝牙通知消息
 * @param notifyCharacteristicsId 通知的服务Id
 * @param deviceId
 * @param serviceId
 */
 export function subscribeBluetoothNotify(notifyCharacteristicsId, deviceId, serviceId, callback) {
  console.log("订阅通知特征值 notifyCharacteristicsId=", notifyCharacteristicsId, "deviceId=",deviceId, "serviceId=",serviceId)
  const _this = this;

  wx.notifyBLECharacteristicValueChange({
    state: true, // 启用 notify 功能
    // 这里的 deviceId 需要已经通过 createBLEConnection 与对应设备建立链接
    deviceId: deviceId,
    // 这里的 serviceId 需要在上面的 getBLEDeviceServices 接口中获取
    serviceId: serviceId,
    // 这里的 characteristicId 需要在上面的 getBLEDeviceCharacteristics 接口中获取
    characteristicId: notifyCharacteristicsId,  //第一步 开启监听 notityid  第二步发送指令 write
    success: function (res) {
      console.log('BLEUtils 订阅通知使能成功, success:', res);

      // 设备返回的方法
      wx.onBLECharacteristicValueChange(function (res) {
        // 此时可以拿到蓝牙设备返回来的数据是一个ArrayBuffer类型数据，
        // 所以需要通过一个方法转换成字符串
        // console.log("watchBluetoothNotify订阅通知回调,res.value=", res.value)
        // 将buff流转为16进制
        const nonceId = ab2hex(res.value)
        console.log(`✅BLEUtils 通知16进制解析：${nonceId}`)
        callback(nonceId)
      })
    },
    fail: function(err) {
      console.log(`err=`, err)
    }
  })
}

/**
 * 通过微信小程序向蓝牙模块发送命令,将指令写入到蓝牙设备当中
 * @param hex 16进制数据
 * @param deviceId  已连接的设备ID
 * @param serviceId 可操作的服务ID
 * @param characteristicId 可写的UUID
 */
 export async function SendBleCmdHex(hex, deviceId, serviceId, characteristicId) {
  const that = this
  console.log(`❤️❤️发送蓝牙信息 >>> 
      hex=${hex}，
      deviceId=${deviceId}，
      serviceId=${serviceId}，
      characteristicId=${characteristicId}
  `)

  return new Promise((resolve, reject) => {
    if (hex.length < 1 || deviceId.length < 1 || serviceId.length < 1 || characteristicId.length < 1) {
      console.log(`Error 缺少参数！`)
      reject(`Error 缺少参数！`);
    } else {
      wx.writeBLECharacteristicValue({
        // 这里的 deviceId 需要在上面的 getBluetoothDevices 或 onBluetoothDeviceFound 接口中获取
        deviceId: deviceId,
        // 这里的 serviceId 需要在上面的 getBLEDeviceServices 接口中获取
        serviceId: serviceId,
        // 这里的 characteristicId 需要在上面的 getBLEDeviceCharacteristics 接口中获取
        characteristicId: characteristicId,//第二步写入的特征值
        // 这里的value是ArrayBuffer类型
        value: hex2buffer(hex),
        success: function (res) {
          console.log(`写入成功，errCode=${res.errCode}，errMsg=${res.errMsg}`)
          resolve(res);
        },
        fail: function (err) {
          console.log(`写入失败，errCode=${err.errCode}，errMsg=${err.errMsg}`)
          reject(err);
        },
        complete:function(){
          // console.log("调用结束");
        }
      })
    }
  })
}
